/**
 * i-net software provides programming examples for illustration only, without warranty
 * either expressed or implied, including, but not limited to, the implied warranties
 * of merchantability and/or fitness for a particular purpose. This programming example
 * assumes that you are familiar with the programming language being demonstrated and
 * the tools used to create and debug procedures. i-net software support professionals
 * can help explain the functionality of a particular procedure, but they will not modify
 * these examples to provide added functionality or construct procedures to meet your
 * specific needs.
 *
 * Copyright © 1999-2026 i-net software GmbH, Berlin, Germany.
**/
package jdbc;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;

import com.inet.report.*;
import com.inet.report.database.BaseDataFactory;
import com.inet.report.database.ColumnInfo;
import com.inet.report.database.TableData;
import com.inet.report.database.TableSourceInfo;

/**
 * This sample shows how to define a Database class that can be used for report design and report execution. This class
 * defines one table with three string columns, that will be shown in i-net Designer - Database Wizard. This sample also
 * demonstrates the usage of the public class com.inet.report.DatabaseMetaDataFactory that is helpful to create meta
 * data Result Sets expected from i-net Designer. To make this class usable by i-net Designer you have to compile it,
 * add it to class path and you have to create a Data Source Configuration with a user-defined driver for this database
 * class. - Compile it with the command: javac -cp .;./core/clearreports.jar
 * samples/jdbc/DatabaseClassUsableInDesigner.java - Adding to class path: To add this class file to the class path, you
 * can create a jar file containing this class (with command
 * "jar cf custom-database-class.jar samples/jdbc/DatabaseClassUsableInDesigner.class") and copying the created jar file
 * into the directory "lib/driver" that is included in the installation directory of i-net Clear Reports or i-net
 * Designer.
 * Create a Data Source Configuration using the Data Source Manager: - select "user-defined driver" - set a name for the
 * Data Source Configuration, e.g. "DesignerDatabaseClass" - set this class "samples.jdbc.DatabaseClassUsableInDesigner"
 * as Database Class for the Data Source Configuration When you design your report in i-net Designer you have to choose
 * the data source "DesignerDatabaseClass" in the Configuration Manager and you'll be able to design a report based on
 * this Database class. In the Database Wizard you can add the table "customers" to the report.
 */
public class DatabaseClassUsableInDesigner extends BaseDataFactory {

    private final static String     TABLE_NAME         = "customers";

    private final static String[]   TABLE_COLUMN_NAMES = new String[] { "customerid", "companyname", "address" };

    private final static int[]      TABLE_COLUMN_TYPES = new int[] { Field.STRING, Field.STRING, Field.STRING };

    private final static Object[][] TABLE_DATA         = new Object[][] { { "ALFKI", "Alfreds Futterkiste", "Obere Str. 57" },
        { "ANATR", "Ana Trujillo Emparedados y helados", "Avda. de la Constitución 2222" } };

    // This effects that the table "customers" will shown in Database Wizard of i-net Designer
    /**
     * {@inheritDoc}
     */
    @Override
    public HashMap<String, TableSourceInfo> getTableSourceInfos( Datasource ds, String catalog ) throws ReportException {
        HashMap<String, TableSourceInfo> result = new HashMap<>();
        result.put( TABLE_NAME, new TableSourceInfo( catalog, null, TABLE_NAME, TableSourceInfo.TYPE_TABLE ) );
        return result;
    }

    // This effects that the used table also has columns
    /**
     * {@inheritDoc}
     */
    @Override
    public List<ColumnInfo> getColumns( Datasource datasource, String catalog, String schema, String name, int type ) throws ReportException {
        if( TABLE_NAME.equalsIgnoreCase( name ) ) {
            ArrayList<ColumnInfo> result = new ArrayList<>();
            for( int i = 0; i < TABLE_COLUMN_NAMES.length; i++ ) {
                result.add( new ColumnInfo( TABLE_COLUMN_NAMES[i], TABLE_COLUMN_TYPES[i] ) );
            }
            return result;
        } else {
            throw new ReportException( "unknown table " + name, 0 );
        }
    }

    // Sets the data for report at runtime
    /**
     * {@inheritDoc}
     */
    @Override
    public TableData getTableSourceData( TableSource ts ) throws ReportException {
        return new TableData( TABLE_COLUMN_NAMES, TABLE_DATA );
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public boolean getReportDataPerInstance() {
        return false;
    }
}
